import { Configuration } from '@crawlee/core';
import type { Log } from '@apify/log';
export interface RecoverableStatePersistenceOptions {
    /**
     * The key under which the state is stored in the KeyValueStore
     */
    persistStateKey: string;
    /**
     * Flag to enable or disable state persistence
     */
    persistenceEnabled?: boolean;
    /**
     * The name of the KeyValueStore to use for persistence.
     * If neither a name nor an id are supplied, the default store will be used.
     */
    persistStateKvsName?: string;
    /**
     * The identifier of the KeyValueStore to use for persistence.
     * If neither a name nor an id are supplied, the default store will be used.
     */
    persistStateKvsId?: string;
}
/**
 * Options for configuring the RecoverableState
 */
export interface RecoverableStateOptions<TStateModel = Record<string, unknown>> extends RecoverableStatePersistenceOptions {
    /**
     * The default state used if no persisted state is found.
     * A deep copy is made each time the state is used.
     */
    defaultState: TStateModel;
    /**
     * A logger instance for logging operations related to state persistence
     */
    logger?: Log;
    /**
     * Configuration instance to use
     */
    config?: Configuration;
    /**
     * Optional function to transform the state to a JSON string before persistence.
     * If not provided, JSON.stringify will be used.
     */
    serialize?: (state: TStateModel) => string;
    /**
     * Optional function to transform a JSON-serialized object back to the state model.
     * If not provided, JSON.parse is used.
     * It is advisable to perform validation in this function and to throw an exception if it fails.
     */
    deserialize?: (serializedState: string) => TStateModel;
}
/**
 * A class for managing persistent recoverable state using a plain JavaScript object.
 *
 * This class facilitates state persistence to a `KeyValueStore`, allowing data to be saved and retrieved
 * across migrations or restarts. It manages the loading, saving, and resetting of state data,
 * with optional persistence capabilities.
 *
 * The state is represented by a plain JavaScript object that can be serialized to and deserialized from JSON.
 * The class automatically hooks into the event system to persist state when needed.
 */
export declare class RecoverableState<TStateModel = Record<string, unknown>> {
    private readonly defaultState;
    private state;
    private readonly persistenceEnabled;
    private readonly persistStateKey;
    private readonly persistStateKvsName?;
    private readonly persistStateKvsId?;
    private keyValueStore;
    private readonly log;
    private readonly config;
    private readonly serialize;
    private readonly deserialize;
    /**
     * Initialize a new recoverable state object.
     *
     * @param options Configuration options for the recoverable state
     */
    constructor(options: RecoverableStateOptions<TStateModel>);
    /**
     * Initialize the recoverable state.
     *
     * This method must be called before using the recoverable state. It loads the saved state
     * if persistence is enabled and registers the object to listen for PERSIST_STATE events.
     *
     * @returns The loaded state object
     */
    initialize(): Promise<TStateModel>;
    /**
     * Clean up resources used by the recoverable state.
     *
     * If persistence is enabled, this method deregisters the object from PERSIST_STATE events
     * and persists the current state one last time.
     */
    teardown(): Promise<void>;
    /**
     * Get the current state.
     */
    get currentValue(): TStateModel;
    /**
     * Reset the state to the default values and clear any persisted state.
     *
     * Resets the current state to the default state and, if persistence is enabled,
     * clears the persisted state from the KeyValueStore.
     */
    reset(): Promise<void>;
    /**
     * Persist the current state to the KeyValueStore.
     *
     * This method is typically called in response to a PERSIST_STATE event, but can also be called
     * directly when needed.
     *
     * @param eventData Optional data associated with a PERSIST_STATE event
     */
    persistState(eventData?: {
        isMigrating: boolean;
    }): Promise<void>;
    /**
     * Load the saved state from the KeyValueStore
     */
    private loadSavedState;
}
//# sourceMappingURL=recoverable_state.d.ts.map