import { Buffer as Buffer$1 } from 'buffer';
import { CookieJar } from 'tough-cookie';
import { EventEmitter } from 'events';
import { AxiosRequestConfig, AxiosResponse, Axios } from 'axios';

/**
 * Curl 类，包装 libcurl 基础功能
 */
declare class Curl {
    private handle;
    private callbackRefs;
    private slists;
    private nextSlistId;
    /**
     * 创建新的 Curl 实例
     */
    constructor();
    /**
     * 关闭并清理 CURL 句柄
     */
    close(): void;
    /**
     * 重置 CURL 句柄为初始状态
     */
    reset(): void;
    /**
     * 设置 CURL 选项
     * @param option CURLOPT 选项
     * @param value 选项值
     */
    setopt(option: number, value: any): void;
    /**
     * 执行 HTTP 请求
     * @returns 错误码
     */
    perform(): number;
    impersonate(impersonate: string, isDefaultHeader?: boolean): any;
    /**
     * 获取 CURL 信息
     * @param info 信息类型
     * @returns 信息值
     */
    getinfo(info: number): any;
    setHeaders(headers: {
        [key: string]: string;
    }): void;
    upkeep(): void;
    static version(): string;
    /**
     * 复制当前 CURL 句柄
     * @returns 新的 Curl 实例
     */
    duplicate(): Curl;
    /**
     * 返回最后一个错误的描述
     * @param code 错误码
     * @returns 错误描述
     */
    static strerror(code: number): string;
    /**
     * 设置 JA3 指纹
     * @param ja3 JA3 指纹字符串
     * @param permute 是否允许扩展乱序
     */
    setJa3Fingerprint(ja3: string, permute?: boolean): void;
    /**
     * 设置 Akamai 指纹
     * @param akamai Akamai 指纹字符串
     */
    setAkamaiFingerprint(akamai: string): void;
    /**
     * 根据扩展 ID 切换扩展
     * @param extensionIds 扩展 ID 集合
     */
    private toggleExtensionsByIds;
    /**
     * 切换单个扩展
     * @param extensionId 扩展 ID
     * @param enable 是否启用
     */
    private toggleExtension;
    /**
     * 设置额外的指纹参数
     * @param extraFp 额外指纹配置
     */
    setExtraFingerprint(extraFp: {
        tlsMinVersion?: number;
        tlsGrease?: boolean;
        tlsPermuteExtensions?: boolean;
        tlsCertCompression?: 'zlib' | 'brotli';
        tlsSignatureAlgorithms?: string[];
        http2StreamWeight?: number;
        http2StreamExclusive?: number;
    }): void;
}

/**
 * libcurl 常量定义
 * 基于 curl_cffi 项目的 const.py
 */
declare enum CURLOPT {
    WRITEDATA = 10001,
    URL = 10002,
    PORT = 3,
    PROXY = 10004,
    USERPWD = 10005,
    PROXYUSERPWD = 10006,
    RANGE = 10007,
    READDATA = 10009,
    ERRORBUFFER = 10010,
    WRITEFUNCTION = 20011,
    READFUNCTION = 20012,
    TIMEOUT = 13,
    INFILESIZE = 14,
    POSTFIELDS = 10015,
    REFERER = 10016,
    FTPPORT = 10017,
    USERAGENT = 10018,
    LOW_SPEED_LIMIT = 19,
    LOW_SPEED_TIME = 20,
    RESUME_FROM = 21,
    COOKIE = 10022,
    HTTPHEADER = 10023,
    HTTPPOST = 10024,
    SSLCERT = 10025,
    KEYPASSWD = 10026,
    CRLF = 27,
    QUOTE = 10028,
    HEADERDATA = 10029,
    COOKIEFILE = 10031,
    SSLVERSION = 32,
    TIMECONDITION = 33,
    TIMEVALUE = 34,
    CUSTOMREQUEST = 10036,
    STDERR = 10037,
    POSTQUOTE = 10039,
    VERBOSE = 41,
    HEADER = 42,
    NOPROGRESS = 43,
    NOBODY = 44,
    FAILONERROR = 45,
    UPLOAD = 46,
    POST = 47,
    DIRLISTONLY = 48,
    APPEND = 50,
    NETRC = 51,
    FOLLOWLOCATION = 52,
    TRANSFERTEXT = 53,
    PUT = 54,
    PROGRESSFUNCTION = 20056,
    XFERINFODATA = 10057,
    AUTOREFERER = 58,
    PROXYPORT = 59,
    POSTFIELDSIZE = 60,
    HTTPPROXYTUNNEL = 61,
    INTERFACE = 10062,
    KRBLEVEL = 10063,
    SSL_VERIFYPEER = 64,
    CAINFO = 10065,
    MAXREDIRS = 68,
    FILETIME = 69,
    TELNETOPTIONS = 10070,
    MAXCONNECTS = 71,
    FRESH_CONNECT = 74,
    FORBID_REUSE = 75,
    RANDOM_FILE = 10076,
    EGDSOCKET = 10077,
    CONNECTTIMEOUT = 78,
    HEADERFUNCTION = 20079,
    HTTPGET = 80,
    SSL_VERIFYHOST = 81,
    COOKIEJAR = 10082,
    SSL_CIPHER_LIST = 10083,
    HTTP_VERSION = 84,
    FTP_USE_EPSV = 85,
    SSLCERTTYPE = 10086,
    SSLKEY = 10087,
    SSLKEYTYPE = 10088,
    SSLENGINE = 10089,
    SSLENGINE_DEFAULT = 90,
    DNS_USE_GLOBAL_CACHE = 91,
    DNS_CACHE_TIMEOUT = 92,
    PREQUOTE = 10093,
    DEBUGFUNCTION = 20094,
    DEBUGDATA = 10095,
    COOKIESESSION = 96,
    CAPATH = 10097,
    BUFFERSIZE = 98,
    NOSIGNAL = 99,
    SHARE = 10100,
    PROXYTYPE = 101,
    ACCEPT_ENCODING = 10102,
    PRIVATE = 10103,
    HTTP200ALIASES = 10104,
    UNRESTRICTED_AUTH = 105,
    FTP_USE_EPRT = 106,
    HTTPAUTH = 107,
    SSL_CTX_FUNCTION = 20108,
    SSL_CTX_DATA = 10109,
    FTP_CREATE_MISSING_DIRS = 110,
    PROXYAUTH = 111,
    SERVER_RESPONSE_TIMEOUT = 112,
    IPRESOLVE = 113,
    MAXFILESIZE = 114,
    INFILESIZE_LARGE = 30115,
    RESUME_FROM_LARGE = 30116,
    MAXFILESIZE_LARGE = 30117,
    NETRC_FILE = 10118,
    USE_SSL = 119,
    POSTFIELDSIZE_LARGE = 30120,
    TCP_NODELAY = 121,
    FTPSSLAUTH = 129,
    IOCTLFUNCTION = 20130,
    IOCTLDATA = 10131,
    FTP_ACCOUNT = 10134,
    COOKIELIST = 10135,
    IGNORE_CONTENT_LENGTH = 136,
    FTP_SKIP_PASV_IP = 137,
    FTP_FILEMETHOD = 138,
    LOCALPORT = 139,
    LOCALPORTRANGE = 140,
    CONNECT_ONLY = 141,
    CONV_FROM_NETWORK_FUNCTION = 20142,
    CONV_TO_NETWORK_FUNCTION = 20143,
    CONV_FROM_UTF8_FUNCTION = 20144,
    MAX_SEND_SPEED_LARGE = 30145,
    MAX_RECV_SPEED_LARGE = 30146,
    FTP_ALTERNATIVE_TO_USER = 10147,
    SOCKOPTFUNCTION = 20148,
    SOCKOPTDATA = 10149,
    SSL_SESSIONID_CACHE = 150,
    SSH_AUTH_TYPES = 151,
    SSH_PUBLIC_KEYFILE = 10152,
    SSH_PRIVATE_KEYFILE = 10153,
    FTP_SSL_CCC = 154,
    TIMEOUT_MS = 155,
    CONNECTTIMEOUT_MS = 156,
    HTTP_TRANSFER_DECODING = 157,
    HTTP_CONTENT_DECODING = 158,
    NEW_FILE_PERMS = 159,
    NEW_DIRECTORY_PERMS = 160,
    POSTREDIR = 161,
    SSH_HOST_PUBLIC_KEY_MD5 = 10162,
    OPENSOCKETFUNCTION = 20163,
    OPENSOCKETDATA = 10164,
    COPYPOSTFIELDS = 10165,
    PROXY_TRANSFER_MODE = 166,
    SEEKFUNCTION = 20167,
    SEEKDATA = 10168,
    CRLFILE = 10169,
    ISSUERCERT = 10170,
    ADDRESS_SCOPE = 171,
    CERTINFO = 172,
    USERNAME = 10173,
    PASSWORD = 10174,
    PROXYUSERNAME = 10175,
    PROXYPASSWORD = 10176,
    NOPROXY = 10177,
    TFTP_BLKSIZE = 178,
    SOCKS5_GSSAPI_SERVICE = 10179,
    SOCKS5_GSSAPI_NEC = 180,
    PROTOCOLS = 181,
    REDIR_PROTOCOLS = 182,
    SSH_KNOWNHOSTS = 10183,
    SSH_KEYFUNCTION = 20184,
    SSH_KEYDATA = 10185,
    MAIL_FROM = 10186,
    MAIL_RCPT = 10187,
    FTP_USE_PRET = 188,
    RTSP_REQUEST = 189,
    RTSP_SESSION_ID = 10190,
    RTSP_STREAM_URI = 10191,
    RTSP_TRANSPORT = 10192,
    RTSP_CLIENT_CSEQ = 193,
    RTSP_SERVER_CSEQ = 194,
    INTERLEAVEDATA = 10195,
    INTERLEAVEFUNCTION = 20196,
    WILDCARDMATCH = 197,
    CHUNK_BGN_FUNCTION = 20198,
    CHUNK_END_FUNCTION = 20199,
    FNMATCH_FUNCTION = 20200,
    CHUNK_DATA = 10201,
    FNMATCH_DATA = 10202,
    RESOLVE = 10203,
    TLSAUTH_USERNAME = 10204,
    TLSAUTH_PASSWORD = 10205,
    TLSAUTH_TYPE = 10206,
    TRANSFER_ENCODING = 207,
    CLOSESOCKETFUNCTION = 20208,
    CLOSESOCKETDATA = 10209,
    GSSAPI_DELEGATION = 210,
    DNS_SERVERS = 10211,
    ACCEPTTIMEOUT_MS = 212,
    TCP_KEEPALIVE = 213,
    TCP_KEEPIDLE = 214,
    TCP_KEEPINTVL = 215,
    SSL_OPTIONS = 216,
    MAIL_AUTH = 10217,
    SASL_IR = 218,
    XFERINFOFUNCTION = 20219,
    XOAUTH2_BEARER = 10220,
    DNS_INTERFACE = 10221,
    DNS_LOCAL_IP4 = 10222,
    DNS_LOCAL_IP6 = 10223,
    LOGIN_OPTIONS = 10224,
    SSL_ENABLE_NPN = 225,
    SSL_ENABLE_ALPN = 226,
    EXPECT_100_TIMEOUT_MS = 227,
    PROXYHEADER = 10228,
    HEADEROPT = 229,
    PINNEDPUBLICKEY = 10230,
    UNIX_SOCKET_PATH = 10231,
    SSL_VERIFYSTATUS = 232,
    SSL_FALSESTART = 233,
    PATH_AS_IS = 234,
    PROXY_SERVICE_NAME = 10235,
    SERVICE_NAME = 10236,
    PIPEWAIT = 237,
    DEFAULT_PROTOCOL = 10238,
    STREAM_WEIGHT = 239,
    STREAM_DEPENDS = 10240,
    STREAM_DEPENDS_E = 10241,
    TFTP_NO_OPTIONS = 242,
    CONNECT_TO = 10243,
    TCP_FASTOPEN = 244,
    KEEP_SENDING_ON_ERROR = 245,
    PROXY_CAINFO = 10246,
    PROXY_CAPATH = 10247,
    PROXY_SSL_VERIFYPEER = 248,
    PROXY_SSL_VERIFYHOST = 249,
    PROXY_SSLVERSION = 250,
    PROXY_TLSAUTH_USERNAME = 10251,
    PROXY_TLSAUTH_PASSWORD = 10252,
    PROXY_TLSAUTH_TYPE = 10253,
    PROXY_SSLCERT = 10254,
    PROXY_SSLCERTTYPE = 10255,
    PROXY_SSLKEY = 10256,
    PROXY_SSLKEYTYPE = 10257,
    PROXY_KEYPASSWD = 10258,
    PROXY_SSL_CIPHER_LIST = 10259,
    PROXY_CRLFILE = 10260,
    PROXY_SSL_OPTIONS = 261,
    PRE_PROXY = 10262,
    PROXY_PINNEDPUBLICKEY = 10263,
    ABSTRACT_UNIX_SOCKET = 10264,
    SUPPRESS_CONNECT_HEADERS = 265,
    REQUEST_TARGET = 10266,
    SOCKS5_AUTH = 267,
    SSH_COMPRESSION = 268,
    MIMEPOST = 10269,
    TIMEVALUE_LARGE = 30270,
    HAPPY_EYEBALLS_TIMEOUT_MS = 271,
    RESOLVER_START_FUNCTION = 20272,
    RESOLVER_START_DATA = 10273,
    HAPROXYPROTOCOL = 274,
    DNS_SHUFFLE_ADDRESSES = 275,
    TLS13_CIPHERS = 10276,
    PROXY_TLS13_CIPHERS = 10277,
    DISALLOW_USERNAME_IN_URL = 278,
    DOH_URL = 10279,
    UPLOAD_BUFFERSIZE = 280,
    UPKEEP_INTERVAL_MS = 281,
    CURLU = 10282,
    TRAILERFUNCTION = 20283,
    TRAILERDATA = 10284,
    HTTP09_ALLOWED = 285,
    ALTSVC_CTRL = 286,
    ALTSVC = 10287,
    MAXAGE_CONN = 288,
    SASL_AUTHZID = 10289,
    MAIL_RCPT_ALLOWFAILS = 290,
    SSLCERT_BLOB = 40291,
    SSLKEY_BLOB = 40292,
    PROXY_SSLCERT_BLOB = 40293,
    PROXY_SSLKEY_BLOB = 40294,
    ISSUERCERT_BLOB = 40295,
    PROXY_ISSUERCERT = 10296,
    PROXY_ISSUERCERT_BLOB = 40297,
    SSL_EC_CURVES = 10298,
    HSTS_CTRL = 299,
    HSTS = 10300,
    HSTSREADFUNCTION = 20301,
    HSTSREADDATA = 10302,
    HSTSWRITEFUNCTION = 20303,
    HSTSWRITEDATA = 10304,
    AWS_SIGV4 = 10305,
    DOH_SSL_VERIFYPEER = 306,
    DOH_SSL_VERIFYHOST = 307,
    DOH_SSL_VERIFYSTATUS = 308,
    CAINFO_BLOB = 40309,
    PROXY_CAINFO_BLOB = 40310,
    SSH_HOST_PUBLIC_KEY_SHA256 = 10311,
    PREREQFUNCTION = 20312,
    PREREQDATA = 10313,
    MAXLIFETIME_CONN = 314,
    MIME_OPTIONS = 315,
    SSH_HOSTKEYFUNCTION = 20316,
    SSH_HOSTKEYDATA = 10317,
    PROTOCOLS_STR = 10318,
    REDIR_PROTOCOLS_STR = 10319,
    WS_OPTIONS = 320,
    CA_CACHE_TIMEOUT = 321,
    QUICK_EXIT = 322,
    HAPROXY_CLIENT_IP = 10323,
    SERVER_RESPONSE_TIMEOUT_MS = 324,
    ECH = 10325,
    TCP_KEEPCNT = 326,
    UPLOAD_FLAGS = 327,
    HTTPBASEHEADER = 11000,
    SSL_SIG_HASH_ALGS = 11001,
    SSL_ENABLE_ALPS = 1002,
    SSL_CERT_COMPRESSION = 11003,
    SSL_ENABLE_TICKET = 1004,
    HTTP2_PSEUDO_HEADERS_ORDER = 11005,
    HTTP2_SETTINGS = 11006,
    SSL_PERMUTE_EXTENSIONS = 1007,
    HTTP2_WINDOW_UPDATE = 1008,
    HTTP2_STREAMS = 11010,
    TLS_GREASE = 1011,
    TLS_EXTENSION_ORDER = 11012,
    STREAM_EXCLUSIVE = 1013,
    TLS_KEY_USAGE_NO_CHECK = 1014,
    TLS_SIGNED_CERT_TIMESTAMPS = 1015,
    TLS_STATUS_REQUEST = 1016,
    TLS_DELEGATED_CREDENTIALS = 11017,
    TLS_RECORD_SIZE_LIMIT = 1018,
    TLS_KEY_SHARES_LIMIT = 1019,
    TLS_USE_NEW_ALPS_CODEPOINT = 1020,
    TLS_USE_FIREFOX_TLS13_CIPHERS = 1021
}
declare const CURLINFO: {
    TEXT: number;
    EFFECTIVE_URL: number;
    RESPONSE_CODE: number;
    TOTAL_TIME: number;
    NAMELOOKUP_TIME: number;
    CONNECT_TIME: number;
    PRETRANSFER_TIME: number;
    SIZE_UPLOAD_T: number;
    SIZE_DOWNLOAD_T: number;
    SPEED_DOWNLOAD_T: number;
    SPEED_UPLOAD_T: number;
    HEADER_SIZE: number;
    REQUEST_SIZE: number;
    SSL_VERIFYRESULT: number;
    FILETIME: number;
    FILETIME_T: number;
    CONTENT_LENGTH_DOWNLOAD_T: number;
    CONTENT_LENGTH_UPLOAD_T: number;
    STARTTRANSFER_TIME: number;
    CONTENT_TYPE: number;
    REDIRECT_TIME: number;
    REDIRECT_COUNT: number;
    PRIVATE: number;
    HTTP_CONNECTCODE: number;
    HTTPAUTH_AVAIL: number;
    PROXYAUTH_AVAIL: number;
    OS_ERRNO: number;
    NUM_CONNECTS: number;
    SSL_ENGINES: number;
    COOKIELIST: number;
    FTP_ENTRY_PATH: number;
    REDIRECT_URL: number;
    PRIMARY_IP: number;
    APPCONNECT_TIME: number;
    CERTINFO: number;
    CONDITION_UNMET: number;
    RTSP_SESSION_ID: number;
    RTSP_CLIENT_CSEQ: number;
    RTSP_SERVER_CSEQ: number;
    RTSP_CSEQ_RECV: number;
    PRIMARY_PORT: number;
    LOCAL_IP: number;
    LOCAL_PORT: number;
    ACTIVESOCKET: number;
    TLS_SSL_PTR: number;
    HTTP_VERSION: number;
    PROXY_SSL_VERIFYRESULT: number;
    SCHEME: number;
    TOTAL_TIME_T: number;
    NAMELOOKUP_TIME_T: number;
    CONNECT_TIME_T: number;
    PRETRANSFER_TIME_T: number;
    STARTTRANSFER_TIME_T: number;
    REDIRECT_TIME_T: number;
    APPCONNECT_TIME_T: number;
    RETRY_AFTER: number;
    EFFECTIVE_METHOD: number;
    PROXY_ERROR: number;
    REFERER: number;
    CAINFO: number;
    CAPATH: number;
    XFER_ID: number;
    CONN_ID: number;
    QUEUE_TIME_T: number;
    USED_PROXY: number;
    POSTTRANSFER_TIME_T: number;
    EARLYDATA_SENT_T: number;
    HTTPAUTH_USED: number;
    PROXYAUTH_USED: number;
    LASTONE: number;
};
declare enum CURL_HTTP_VERSION {
    NONE = 0,
    V1_0 = 1,// please use HTTP 1.0 in the request */
    V1_1 = 2,// please use HTTP 1.1 in the request */
    V2_0 = 3,// please use HTTP 2 in the request */
    V2TLS = 4,// use version 2 for HTTPS, version 1.1 for HTTP */
    V2_PRIOR_KNOWLEDGE = 5,// please use HTTP 2 without HTTP/1.1 Upgrade */
    V3 = 30,// Makes use of explicit HTTP/3 with fallback.
    V3ONLY = 31
}
declare const CURLE: {
    OK: number;
    UNSUPPORTED_PROTOCOL: number;
    FAILED_INIT: number;
    URL_MALFORMAT: number;
    NOT_BUILT_IN: number;
    COULDNT_RESOLVE_PROXY: number;
    COULDNT_RESOLVE_HOST: number;
    COULDNT_CONNECT: number;
    WEIRD_SERVER_REPLY: number;
    REMOTE_ACCESS_DENIED: number;
    FTP_ACCEPT_FAILED: number;
    FTP_WEIRD_PASS_REPLY: number;
    FTP_ACCEPT_TIMEOUT: number;
    FTP_WEIRD_PASV_REPLY: number;
    FTP_WEIRD_227_FORMAT: number;
    FTP_CANT_GET_HOST: number;
    HTTP2: number;
    FTP_COULDNT_SET_TYPE: number;
    PARTIAL_FILE: number;
    FTP_COULDNT_RETR_FILE: number;
    OBSOLETE20: number;
    QUOTE_ERROR: number;
    HTTP_RETURNED_ERROR: number;
    WRITE_ERROR: number;
    OBSOLETE24: number;
    UPLOAD_FAILED: number;
    READ_ERROR: number;
    OUT_OF_MEMORY: number;
    OPERATION_TIMEDOUT: number;
    FTP_PORT_FAILED: number;
    FTP_COULDNT_USE_REST: number;
    OBSOLETE32: number;
    RANGE_ERROR: number;
    HTTP_POST_ERROR: number;
    SSL_CONNECT_ERROR: number;
    BAD_DOWNLOAD_RESUME: number;
    FILE_COULDNT_READ_FILE: number;
    LDAP_CANNOT_BIND: number;
    LDAP_SEARCH_FAILED: number;
    OBSOLETE40: number;
    FUNCTION_NOT_FOUND: number;
    ABORTED_BY_CALLBACK: number;
    BAD_FUNCTION_ARGUMENT: number;
    OBSOLETE44: number;
    INTERFACE_FAILED: number;
    OBSOLETE46: number;
    TOO_MANY_REDIRECTS: number;
    UNKNOWN_OPTION: number;
    TELNET_OPTION_SYNTAX: number;
    OBSOLETE50: number;
    PEER_FAILED_VERIFICATION: number;
    GOT_NOTHING: number;
    SSL_ENGINE_NOTFOUND: number;
    SSL_ENGINE_SETFAILED: number;
    SEND_ERROR: number;
    RECV_ERROR: number;
    OBSOLETE57: number;
    SSL_CERTPROBLEM: number;
    SSL_CIPHER: number;
    SSL_CACERT: number;
    BAD_CONTENT_ENCODING: number;
    LDAP_INVALID_URL: number;
    FILESIZE_EXCEEDED: number;
    USE_SSL_FAILED: number;
    SEND_FAIL_REWIND: number;
    SSL_ENGINE_INITFAILED: number;
    LOGIN_DENIED: number;
    TFTP_NOTFOUND: number;
    TFTP_PERM: number;
    REMOTE_DISK_FULL: number;
    TFTP_ILLEGAL: number;
    TFTP_UNKNOWNID: number;
    REMOTE_FILE_EXISTS: number;
    TFTP_NOSUCHUSER: number;
    CONV_FAILED: number;
    CONV_REQD: number;
    SSL_CACERT_BADFILE: number;
    REMOTE_FILE_NOT_FOUND: number;
    SSH: number;
    SSL_SHUTDOWN_FAILED: number;
    AGAIN: number;
    SSL_CRL_BADFILE: number;
    SSL_ISSUER_ERROR: number;
    FTP_PRET_FAILED: number;
    RTSP_CSEQ_ERROR: number;
    RTSP_SESSION_ERROR: number;
    FTP_BAD_FILE_LIST: number;
    CHUNK_FAILED: number;
    NO_CONNECTION_AVAILABLE: number;
    SSL_PINNEDPUBKEYNOTMATCH: number;
    SSL_INVALIDCERTSTATUS: number;
    HTTP2_STREAM: number;
    RECURSIVE_API_CALL: number;
    AUTH_ERROR: number;
    HTTP3: number;
    QUIC_CONNECT_ERROR: number;
    PROXY: number;
    SSL_CLIENTCERT: number;
    UNRECOVERABLE_POLL: number;
};
declare enum CURL_SSLVERSION {
    DEFAULT = 0,
    TLSv1 = 1,
    SSLv2 = 2,
    SSLv3 = 3,
    TLSv1_0 = 4,
    TLSv1_1 = 5,
    TLSv1_2 = 6,
    TLSv1_3 = 7,
    MAX_DEFAULT = 65536
}
declare enum CURL_IP_RESOLVE {
    WHATEVER = 0,
    V4 = 1,
    V6 = 2
}
type CURL_IMPERSONATE = "edge99" | "edge101" | "chrome99" | "chrome100" | "chrome101" | "chrome104" | "chrome107" | "chrome110" | "chrome116" | "chrome119" | "chrome120" | "chrome123" | "chrome124" | "chrome131" | "chrome133a" | "chrome136" | "chrome99_android" | "chrome131_android" | "safari153" | "safari155" | "safari170" | "safari172_ios" | "safari180" | "safari180_ios" | "safari184" | "safari184_ios" | "firefox133" | "firefox135" | "tor145" | "chrome" | "edge" | "safari" | "safari_ios" | "chrome_android" | "firefox" | "safari15_3" | "safari15_5" | "safari17_0" | "safari17_2_ios" | "safari18_0" | "safari18_0_ios" | "safari18_4" | "safari18_4_ios";

declare const _constants_CURLE: typeof CURLE;
declare const _constants_CURLINFO: typeof CURLINFO;
type _constants_CURLOPT = CURLOPT;
declare const _constants_CURLOPT: typeof CURLOPT;
type _constants_CURL_HTTP_VERSION = CURL_HTTP_VERSION;
declare const _constants_CURL_HTTP_VERSION: typeof CURL_HTTP_VERSION;
type _constants_CURL_IMPERSONATE = CURL_IMPERSONATE;
type _constants_CURL_IP_RESOLVE = CURL_IP_RESOLVE;
declare const _constants_CURL_IP_RESOLVE: typeof CURL_IP_RESOLVE;
type _constants_CURL_SSLVERSION = CURL_SSLVERSION;
declare const _constants_CURL_SSLVERSION: typeof CURL_SSLVERSION;
declare namespace _constants {
  export { _constants_CURLE as CURLE, _constants_CURLINFO as CURLINFO, _constants_CURLOPT as CURLOPT, _constants_CURL_HTTP_VERSION as CURL_HTTP_VERSION, type _constants_CURL_IMPERSONATE as CURL_IMPERSONATE, _constants_CURL_IP_RESOLVE as CURL_IP_RESOLVE, _constants_CURL_SSLVERSION as CURL_SSLVERSION };
}

declare class ResponseHeader {
    header: {
        [key: string]: Array<string>;
    };
    constructor();
    set(key: string, value: string): void;
    get(key: string): string | undefined;
    getAll(key: string): string[] | undefined;
    toString(): string;
}
type RequestHeader = {
    [key: string]: string;
};
/**
 * HTTP 请求选项接口
 */
interface RequestOptions {
    url: string;
    method?: 'GET' | 'POST' | 'PUT' | 'DELETE' | 'PATCH' | 'HEAD' | 'OPTIONS';
    headers?: RequestHeader;
    data?: any;
    params?: {
        [key: string]: string | number;
    };
    timeout?: number;
    followRedirects?: boolean;
    maxRedirects?: number;
    proxy?: string;
    referer?: string;
    acceptEncoding?: string;
    userAgent?: string;
    impersonate?: CURL_IMPERSONATE;
    verifySsl?: boolean;
    jar?: CookieJar;
    ipType?: 'ipv4' | 'ipv6' | 'auto';
    retryCount?: number;
    auth?: {
        username: string;
        password: string;
    };
}
/**
 * HTTP 响应接口
 */
interface Response {
    status: number;
    statusText: string;
    headers: ResponseHeader;
    data: string;
    url: string;
    redirectCount: number;
    buffer: Buffer$1;
    duration: number;
}
declare function request(options: RequestOptions): Promise<Response>;
declare function get(url: string, options?: Omit<RequestOptions, 'url' | 'method'>): Promise<Response>;
/**
 * POST 请求
 */
declare function post(url: string, data?: any, options?: Omit<RequestOptions, 'url' | 'method'>): Promise<Response>;
/**
 * PUT 请求
 */
declare function put(url: string, data?: any, options?: Omit<RequestOptions, 'url' | 'method'>): Promise<Response>;
/**
 * DELETE 请求
 */
declare function apiDelete(url: string, options?: Omit<RequestOptions, 'url' | 'method'>): Promise<Response>;
/**
 * PATCH 请求
 */
declare function patch(url: string, data?: any, options?: Omit<RequestOptions, 'url' | 'method'>): Promise<Response>;

declare const _default: {
    request: typeof request;
    get: typeof get;
    post: typeof post;
    put: typeof put;
    delete: typeof apiDelete;
    patch: typeof patch;
    Response: {
        new (body?: BodyInit | null, init?: ResponseInit): globalThis.Response;
        prototype: globalThis.Response;
        error(): globalThis.Response;
        json(data: any, init?: ResponseInit): globalThis.Response;
        redirect(url: string | URL, status?: number): globalThis.Response;
    };
};

type request$1_RequestOptions = RequestOptions;
type request$1_Response = Response;
type request$1_ResponseHeader = ResponseHeader;
declare const request$1_ResponseHeader: typeof ResponseHeader;
declare const request$1_get: typeof get;
declare const request$1_patch: typeof patch;
declare const request$1_post: typeof post;
declare const request$1_put: typeof put;
declare const request$1_request: typeof request;
declare namespace request$1 {
  export { type request$1_RequestOptions as RequestOptions, type request$1_Response as Response, request$1_ResponseHeader as ResponseHeader, _default as default, apiDelete as delete, request$1_get as get, request$1_patch as patch, request$1_post as post, request$1_put as put, request$1_request as request };
}

/**
 * WebSocket消息类型
 */
declare enum WsMessageType {
    TEXT = 1,
    BINARY = 2,
    CLOSE = 8,
    PING = 9,
    PONG = 10
}
/**
 * WebSocket选项
 */
interface WebSocketOptions {
    headers?: {
        [key: string]: string;
    };
    protocols?: string[];
    timeout?: number;
    pingInterval?: number;
    pongTimeout?: number;
}
/**
 * WebSocket消息
 */
interface WsMessage {
    type: WsMessageType;
    data: Buffer;
    text?: string;
}
/**
 * WebSocket客户端
 */
declare class WebSocket extends EventEmitter {
    private curl;
    private connected;
    private closing;
    private pingTimer?;
    private pongTimer?;
    private options;
    constructor(url: string, options?: WebSocketOptions);
    /**
     * 设置CURL选项
     */
    private setupCurl;
    /**
     * 生成WebSocket密钥
     */
    private generateWebSocketKey;
    /**
     * 连接WebSocket
     */
    connect(): Promise<void>;
    /**
     * 发送文本消息
     */
    sendText(text: string): Promise<void>;
    /**
     * 发送二进制消息
     */
    sendBinary(data: Buffer): Promise<void>;
    /**
     * 发送ping
     */
    ping(data?: Buffer): Promise<void>;
    /**
     * 发送pong
     */
    pong(data?: Buffer): Promise<void>;
    /**
     * 发送WebSocket帧
     */
    private sendFrame;
    /**
     * 构建WebSocket帧
     */
    private buildFrame;
    /**
     * 处理接收到的WebSocket数据
     */
    private handleWebSocketData;
    /**
     * 解析WebSocket帧
     */
    private parseFrame;
    /**
     * 处理消息
     */
    private handleMessage;
    /**
     * 处理pong响应
     */
    private handlePong;
    /**
     * 处理关闭
     */
    private handleClose;
    /**
     * 开始ping间隔
     */
    private startPingInterval;
    /**
     * 停止ping间隔
     */
    private stopPingInterval;
    /**
     * 关闭WebSocket连接
     */
    close(code?: number, reason?: string): Promise<void>;
    /**
     * 获取连接状态
     */
    get readyState(): number;
}

declare const constants: typeof _constants;
declare const libcurlVersion: () => any;

/**
 * 日志工具，支持不同级别的日志输出
 */
declare enum LogLevel {
    ERROR = 0,
    WARN = 1,
    INFO = 2,
    DEBUG = 3
}
declare class Logger {
    private level;
    constructor();
    /**
     * 设置日志级别
     */
    setLevel(level: LogLevel): void;
    /**
     * 获取当前日志级别
     */
    getLevel(): LogLevel;
    /**
     * 错误日志
     */
    error(...args: any[]): void;
    /**
     * 警告日志
     */
    warn(...args: any[]): void;
    /**
     * 信息日志
     */
    info(...args: any[]): void;
    /**
     * 调试日志
     */
    debug(...args: any[]): void;
    /**
     * 检查是否启用了调试模式
     */
    isDebugEnabled(): boolean;
    /**
     * 检查是否启用了警告模式
     */
    isWarnEnabled(): boolean;
}
declare const logger: Logger;

type CurlSessionConfig = Omit<AxiosRequestConfig<any>, 'proxy'> & Omit<RequestOptions, 'url'> & {
    url?: string;
    proxy?: string;
};
type CurlSessionResponse = Omit<AxiosResponse<any>, 'headers'> & Response & {
    text: string;
};
declare class CurlSession {
    jar?: CookieJar;
    axios: Axios;
    private _config?;
    constructor(config?: CurlSessionConfig);
    private initHook;
    clone(): CurlSession;
    post(url: string, data?: any, config?: CurlSessionConfig): Promise<CurlSessionResponse>;
    get(url: string, config?: CurlSessionConfig): Promise<CurlSessionResponse>;
    request(config: CurlSessionConfig): Promise<CurlSessionResponse>;
    delete(url: string, config?: CurlSessionConfig): Promise<CurlSessionResponse>;
    put(url: string, data?: any, config?: CurlSessionConfig): Promise<CurlSessionResponse>;
    patch(url: string, data?: any, config?: CurlSessionConfig): Promise<CurlSessionResponse>;
    head(url: string, config?: CurlSessionConfig): Promise<CurlSessionResponse>;
    options(url: string, config?: CurlSessionConfig): Promise<CurlSessionResponse>;
}

export { Curl, CurlSession, type CurlSessionConfig, type CurlSessionResponse, LogLevel, WebSocket, type WebSocketOptions, type WsMessage, WsMessageType, constants as bindingConstants, _constants as constants, libcurlVersion, logger, request$1 as req };
