const { Curl, CurlSslVersion } = require('node-libcurl')
const Caseless = require('caseless')
const UserAgents = require('user-agents')
const multipart = require('./multipart.js')
const { compress, uncompress, randomIp } = require('./utils.js')
const tough = require('tough-cookie')

function randomCiphers(tlsVersion, minCiphers = 15) {

    const cipherTLSMap = {
        // TLS 1.3 Ciphers
        "TLS_AES_128_GCM_SHA256": 7,
        "TLS_AES_256_GCM_SHA384": 7,
        "TLS_CHACHA20_POLY1305_SHA256": 7,
        "TLS_AES_128_CCM_SHA256": 7,
        "TLS_AES_128_CCM_8_SHA256": 7,

        // TLS 1.2 Ciphers
        "ECDHE-ECDSA-AES128-GCM-SHA256": 6,
        "ECDHE-RSA-AES128-GCM-SHA256": 6,
        "ECDHE-ECDSA-AES256-GCM-SHA384": 6,
        "ECDHE-RSA-AES256-GCM-SHA384": 6,
        "ECDHE-ECDSA-CHACHA20-POLY1305": 6,
        //"ECDHE-RSA-CHACHA20-POLY1305": 6,
        "AES-128-CCM": 6,
        "AES-256-CCM": 6,
        "DH-2048": 6,
        "DH-4096": 6,
        "AES128-GCM-SHA256": 6,
        "AES256-GCM-SHA384": 6,
        "ECDHE-ECDSA-AES128-SHA": 6,
        "ECDHE-RSA-AES128-SHA": 6,
        "ECDHE-ECDSA-AES256-SHA": 6,
        "ECDHE-RSA-AES256-SHA": 6,
        "DHE-RSA-AES128-GCM-SHA256": 6,
        "DHE-RSA-AES256-GCM-SHA384": 6,
        "DHE-RSA-CHACHA20-POLY1305": 6,
        "DHE-DSS-AES128-GCM-SHA256": 6,
        "DHE-DSS-AES256-GCM-SHA384": 6,
        "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256": 6,
        "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256": 6,
        "ECDHE-ECDSA-AES256-CBC-SHA384": 6,
        "ECDHE-RSA-AES256-CBC-SHA384": 6,
        "DHE-RSA-AES256-CBC-SHA256": 6,
        "ECDHE-ECDSA-AES128-CBC-SHA": 6,
        "ECDHE-RSA-AES128-CBC-SHA": 6,
        "ECDHE-ECDSA-AES256-CBC-SHA": 6,
        "ECDHE-RSA-AES256-CBC-SHA": 6,
        "DHE-RSA-AES128-CBC-SHA256": 6,
        "DHE-RSA-AES128-CBC-SHA": 6,
        "DHE-RSA-AES256-CBC-SHA": 6,
        "DHE-DSS-AES128-CBC-SHA256": 6,
        "DHE-DSS-AES256-CBC-SHA256": 6,
        "ECDHE-ECDSA-WITH-CHACHA20-POLY1305": 6,
        "ECDHE-RSA-WITH-CHACHA20-POLY1305": 6,
        "DHE-RSA-WITH-CHACHA20-POLY1305": 6,
        "TLS_RSA_WITH_AES_128_GCM_SHA256": 6,
        "TLS_RSA_WITH_AES_256_GCM_SHA384": 6,
        "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA": 6,
        "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA": 6
    }

    const filteredCiphers = Object.keys(cipherTLSMap).filter(cipher => cipherTLSMap[cipher] === tlsVersion)

    if (filteredCiphers.length === 0) {
        return { error: "Nenhuma cifra encontrada para a versão especificada" }
    }

    let selectedCiphers

    if (tlsVersion === 7) {
        selectedCiphers = filteredCiphers.sort(() => Math.random() - 0.5)
    } else if (tlsVersion === 6) {
        selectedCiphers = []
        const numCiphers = Math.min(minCiphers, filteredCiphers.length)

        while (selectedCiphers.length < numCiphers) {
            const randomCipher = filteredCiphers[Math.floor(Math.random() * filteredCiphers.length)]
            if (!selectedCiphers.includes(randomCipher)) {
                selectedCiphers.push(randomCipher)
            }
        }
    } else {
        return { error: "Versão TLS inválida. Use apenas 'TLS 1.2' ou 'TLS 1.3'." }
    }

    return {
        ciphers: selectedCiphers.join(":"),
        tlsVersion
    }
}


class CurlYoshi {
    constructor(options) {
        return this.init(options)
    }

    static parse = {
        stringify: function (data) {
            try {
                return JSON.stringify(data)
            } catch (error) {
                return data
            }
        },
        json: function (data) {
            try {
                return JSON.parse(data)
            } catch (error) {
                return data
            }
        },
        xmlToJson: function (data) {
            try {
                const json = {}
                for (const xml of data.matchAll(/(?:<(\w*)(?:\s[^>]*)*>)((?:(?!<\1).)*)(?:<\/\1>)|<(\w*)(?:\s*)*\/>/gm)) {
                    const key = xml[1] || xml[3]
                    const value = xml[2] && this.xmlToJson(xml[2])
                    if (json[key] !== undefined) {
                        if (!Array.isArray(json[key])) {
                            json[key] = [json[key]]
                        }
                        json[key].push((value && Object.keys(value).length) ? value : xml[2])
                    } else {
                        json[key] = (value && Object.keys(value).length) ? value : xml[2]
                    }
                }
                return json
            } catch (error) {
                return data
            }
        }
    }

    
    readProtoBuffer(data) {
        let index = 0
        return (buffer, size, count) => {
            const bytesToRead = size * count
            if (index >= data.length) return 0
            const bytesCopied = data.copy(buffer, 0, index, index + bytesToRead)
            index += bytesCopied
            return bytesCopied
        }
    }

    init(options) {
        if (options.qs) {
            const parsed = new URL(options.url)
            options.url += !parsed.search ? `?` : parsed.search == '?' ? '' : '&'
            const values = []
            for (let key in options.qs) {
                values.push(`${key}=${options.qs[key]}`)
            }
            options.url += values.join('&')
            delete options.qs
        }

        return new Promise(resolve => {
            const curl = new Curl()

            curl.setOpt('URL', options.url)

            if (options.verbose) {
                curl.setOpt(Curl.option.VERBOSE, true)
                curl.setOpt(Curl.option.DEBUGFUNCTION, (infoType, content) => {
                    if (infoType == 0) {
                        console.log(Buffer.from(content).toString().trim())
                    }
                })
            }

            curl.setOpt(Curl.option.SSL_VERIFYPEER, false)
            curl.setOpt(Curl.option.SSL_VERIFYHOST, false)
            curl.setOpt(Curl.option.SSL_VERIFYSTATUS, false)

            curl.setOpt(Curl.option.SSL_ENABLE_ALPN, false)
            curl.setOpt(Curl.option.SSL_ENABLE_NPN, false)

            if (options.cert) {
                curl.setOpt(Curl.option.SSLCERT, options.cert)
            }

            if (options.key) {
                curl.setOpt(Curl.option.SSLKEY, options.key)
            }

            curl.setOpt(Curl.option.CUSTOMREQUEST, options.method || 'GET')

            if (options.timeout) {
                curl.setOpt(Curl.option.TIMEOUT_MS, options.timeout)
            }

            if (options.forever) {
                curl.setOpt(Curl.option.TCP_KEEPALIVE, 2)
                curl.setOpt(Curl.option.FORBID_REUSE, 0)
                curl.setOpt(Curl.option.FRESH_CONNECT, 0)
            } else {
                curl.setOpt(Curl.option.TCP_KEEPALIVE, 0)
                curl.setOpt(Curl.option.FORBID_REUSE, 2)
                curl.setOpt(Curl.option.FRESH_CONNECT, 1)
            }

            curl.setOpt(Curl.option.PATH_AS_IS, options.rebuild)

            let headers = {}
            for (let header of Object.keys(options.headers).filter(Boolean)) {
                if (header.toLowerCase() == 'cookie') {
                    if (options.jar) {
                        let cookiesInJar = options.jar.getCookieStringSync(options.url)
                        if (!cookiesInJar) {
                            Object.assign(headers, { [header]: options.headers[header] })
                        } else {
                            Object.assign(headers, { [header]: `${options.headers[header]}; ${cookiesInJar}` })
                        }
                    } else {
                        Object.assign(headers, { [header]: options.headers[header] })
                    }
                } else {
                    Object.assign(headers, { [header]: options.headers[header] })
                }
            }

            let caseless = Caseless(headers)
            if (!caseless.has('cookie') && options.jar) {
                let cookiesInJar = options.jar.getCookieStringSync(options.url)
                if (cookiesInJar) Object.assign(headers, { cookie: cookiesInJar })
            }

            let postdata
            if (options.form) {
                let data = []
                let keys = Object.keys(options.form)

                for (let i in keys) {
                    let key = keys[i]
                    data.push(`${key}=${options.form[key]}`)
                }

                let fields = data.join('&')

                if (!headers) {
                    headers = {
                        'content-type': 'application/x-www-form-urlencoded',
                    }
                }
                if (!caseless.get('content-type')) {
                    caseless.set('content-type', 'application/x-www-form-urlencoded')
                }
                postdata = fields
                delete options.form
            } else if (options.body) {
                if (!headers) {
                    headers = {
                        'content-type': 'application/x-www-form-urlencoded',
                    }
                }
                if (!caseless.get('content-type')) {
                    caseless.set('content-type', 'application/x-www-form-urlencoded')
                }
                postdata = options.body
                delete options.body
            } else if (options.json) {
                if (typeof options.json !== 'boolean') {
                    if (!headers) {
                        headers = {
                            'content-type': 'application/json'
                        }
                    }
                    if (!caseless.get('content-type')) {
                        caseless.set('content-type', 'application/json')
                    }
                    postdata = CurlYoshi.parse.stringify(options.json)
                    delete options.json
                }
            } else if (options.multipart) {
                let boundary = options.boundary || 'X-CURLYOSHI-BOUNDARY'
                if (!headers) {
                    headers = {
                        'content-type': `multipart/form-data; boundary=${boundary}`
                    }
                }
                if (!caseless.get('content-type')) {
                    caseless.set('content-type', `multipart/form-data; boundary=${boundary}`)
                }
                multipart(options.multipart, boundary, function (err, parts) {
                    if (err) throw (err)
                    postdata = parts
                })
                delete options.multipart
            }

            if (typeof postdata == 'string' && !options.gzip) {
                curl.setOpt(Curl.option.POSTFIELDS, postdata)
            } else if (Buffer.isBuffer(postdata) && !options.gzip) {
                curl.setOpt(Curl.option.POST, 1)
                curl.setOpt(Curl.option.POSTFIELDSIZE, postdata.length)
                curl.setOpt(Curl.option.READFUNCTION, this.readProtoBuffer(postdata))
            }

            if (options.gzip) {
                caseless.set('content-encoding', 'gzip')
                let compressed = compress(postdata, 'gzip')
                curl.setOpt(Curl.option.POST, 1)
                curl.setOpt(Curl.option.READFUNCTION, (buffer, size, nmemb) => {
                    const chunkSize = size * nmemb
                    const chunk = compressed.slice(0, chunkSize)
                    compressed = compressed.slice(chunkSize)
                    chunk.copy(buffer)
                    return chunk.length
                })
                curl.setOpt(Curl.option.POSTFIELDSIZE, compressed.length)
                curl.setOpt(Curl.option.ACCEPT_ENCODING, 'gzip, deflate')
            } else {
                curl.setOpt(Curl.option.HTTP_CONTENT_DECODING, '0')
            }

            if (options.http2) {
                curl.setOpt(Curl.option.SSL_ENABLE_ALPN, true)
                curl.setOpt(Curl.option.HTTP_VERSION, 'CURL_HTTP_VERSION_2_0')
            } else {
                curl.setOpt(Curl.option.HTTP_VERSION, 'CURL_HTTP_VERSION_1_1')
            }

            if (options.xff) {
                caseless.set('x-real-ip', randomIp())
                caseless.set('x-forwarded-for', `${randomIp()}, ${randomIp()}, ${randomIp()}`)
                caseless.set('true-client-ip', randomIp())
            }

            if (!caseless.has('user-agent')) {
                let userAgent = new UserAgents({ deviceCategory: options.desktop ? 'desktop' : 'mobile' })
                caseless.set('user-agent', userAgent.toString())
            }
            else
            {
                curl.setOpt(Curl.option.USERAGENT, caseless.get('user-agent'));
            }

            curl.setOpt(Curl.option.HTTPHEADER, Object.keys(headers).map(key => `${key}: ${headers[key]}`))

            if (options.proxy) {
                let proxy = typeof options.proxy == "function" ? options.proxy() : options.proxy
                curl.setOpt(Curl.option.PROXY, proxy)
            }

            // SSL (6 = 1.2)
            let sslversion = (options.ssl) ? options.ssl : 6;
            curl.setOpt(Curl.option.SSLVERSION, sslversion);

            let cipherList;
            if (options.ciphers)
            {
                if (Array.isArray(options.ciphers))
                {
                    curl.setOpt(Curl.option.SSL_CIPHER_LIST, options.ciphers.join(':'));
                }
                else
                {
                    cipherList = randomCiphers(sslversion, sslversion).ciphers
                    curl.setOpt(Curl.option.SSL_CIPHER_LIST, cipherList, options.minCiphers);
                }
            }

            var responseBuffer = Buffer.from([]);

            curl.setOpt(Curl.option.WRITEFUNCTION, (data) => {
                responseBuffer = Buffer.concat([responseBuffer, data]);

                return data.length; 
            });

            curl.on('end', function (statusCode, data, headers) {
                let respHeaders = headers[headers.length - 1]
                delete respHeaders.result
                if (options.jar) {
                    let { protocol, host } = new URL(options.url)
                    let hostname = `${protocol}//${host}/`
                    let caseless = Caseless(respHeaders)

                    let setCookies = caseless.get('set-cookie')

                    if (setCookies) {
                        if (!Array.isArray(setCookies)) {
                            setCookies = [setCookies]
                        }

                        setCookies.forEach(function (line) {
                            try {
                                let cookie = tough.Cookie.parse(line)
                                if (cookie) {
                                    options.jar.setCookieSync(cookie, hostname)
                                }
                            } catch (error) {
                                console.error(error)
                            }
                        })
                    }
                }

                if (options.followRedirects && curl.getInfo(Curl.info.REDIRECT_URL)) {
                    options.redirectCount = options.redirectCount + 1 || 1
                    if (options.redirectCount != options.maxRedirects) {
                        options.forever = true
                        options.url = curl.getInfo(Curl.info.REDIRECT_URL)

                        options.method = options.followMethod || options.method || 'GET'
                        if (options.method.toLowerCase() == 'get') {
                            delete options.form
                            delete options.body
                        }

                        this.close()
                        return resolve(new CurlYoshi(options))
                    }
                }

                data = responseBuffer.toString('utf8').trim();

                if (options.xmlToJson) {
                    data = CurlYoshi.parse.xmlToJson(data)
                }
                else if (options.raw)
                {
                    data = responseBuffer
                }
                else {
                    data = CurlYoshi.parse.json(data)
                }

                ///
                if (statusCode == 200) {
                    console.log(cipherList);
                }

                let response = {
                    body: data,
                    headers: respHeaders,
                    statusCode: statusCode,
                }

                this.close()

                resolve(response)
            })

            curl.on('error', function (error) {
                curl.close.bind(curl)
                this.close()
                resolve(error.message)
            })

            try {
                curl.perform()
            } catch (error) {
                curl.close()
                resolve(error.message)
            }
        })
    }
}

module.exports = CurlYoshi